function draw_prior(X, Y, method, k, p)
% DRAW_PRIOR Visualizes the prior distribution learned by Naive Bayes for two selected features
%   X      : Feature matrix
%   Y      : Class labels
%   method : Naive Bayes distribution type (e.g., 'normal', 'kernel', 'mvmn')
%   k, p   : Indices of two features to be visualized

    % Select only the k-th and p-th columns (features) for visualization
    X = X(:, [k, p]);

    % Train a Naive Bayes model using the selected features
    Mdl = fitcnb(X, Y, 'DistributionNames', method);

    % Plot the data points using a scatter plot, colored by class label
    figure
    gscatter(X(:,1), X(:,2), Y);  % Scatter plot of the two features

    % Save the current axis limits so they can be restored later
    h = gca;
    cxlim = h.XLim;  % Get X-axis limits
    cylim = h.YLim;  % Get Y-axis limits

    hold on  % Hold the current plot

    % Extract Gaussian distribution parameters from the model
    Params = cell2mat(Mdl.DistributionParameters);  % Convert from cell to matrix

    % Extract the means for each class (1 row per class, 2 columns for features)
    Mu = Params(2*(1:3)-1, 1:2);  % Extract mean vectors for each class

    Sigma = zeros(2, 2, 3);  % Initialize covariance matrices (diagonal for simplicity)

    for j = 1:3  % Assuming 3 classes
        % Build diagonal covariance matrix (variances from model)
        Sigma(:,:,j) = diag(Params(2*j, :)).^2;

        % Define the plot range for contour drawing (4 std deviations)
        xlim = Mu(j,1) + 4 * [1 -1] * sqrt(Sigma(1,1,j));
        ylim = Mu(j,2) + 4 * [1 -1] * sqrt(Sigma(2,2,j));

        % Draw contour lines of the multivariate normal distribution for class j
        ezcontour(@(x1, x2) mvnpdf([x1, x2], Mu(j,:), Sigma(:,:,j)), [xlim ylim]);
    end

    % Restore original axis limits
    h.XLim = cxlim;
    h.YLim = cylim;

    % Add labels and title
    title('Naive Bayes Classifier');
    xlabel("x" + num2str(k, '%d'));  % Label x-axis with feature index
    ylabel("x" + num2str(p, '%d'));  % Label y-axis with feature index
    hold off  % Release the plot

    % Construct output path and file name
    root = find_root();
    name = "x" + num2str(k, '%d') + "_" + "x" + num2str(p, '%d') + '.png';
    fn = fullfile(root, 'output_bayes', 'prior', name);

    % Save figure as PNG
    print('-dpng', fn);
    fprintf(' %s \n', fn);
end
